<?php

// Fügen Sie DIESE ZEILE am Anfang Ihrer PHP-Skripte ein:
require __DIR__ . '/config.php'; // Pfad anpassen, falls config.php nicht im selben Ordner ist

// WICHTIG: Die ID des Kalenders, den Sie anzeigen möchten.
// Diesen Wert finden Sie in der Baikal-Tabelle 'calendars'.
define('TARGET_CALENDAR_ID', 1); // BITTE ANPASSEN!

// ----------------------------------------------------
// 2. ERFORDERLICHE BIBLIOTHEKEN & SETUP
// ----------------------------------------------------
// Dies setzt voraus, dass Sie Sabre/VObject (empfohlen) via Composer installiert haben:
// composer require sabre/vobject
require 'vendor/autoload.php'; 

use Sabre\VObject;

// Setzt den Content-Type-Header, da das Frontend JSON-Daten erwartet
header('Content-Type: application/json');

try {
    // 3. Datenbankverbindung herstellen (PDO)
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4", DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 4. Events aus der Datenbank abrufen
    // Wir benötigen 'id' (für spätere Updates) und 'calendardata' (den ICS-Block).
    $stmt = $pdo->prepare("
        SELECT id, calendardata
        FROM calendarobjects
        WHERE calendarid = :calendar_id
    ");
    $stmt->bindParam(':calendar_id', TARGET_CALENDAR_ID);
    $stmt->execute();
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $fullCalendarEvents = [];

    // 5. ICS-Daten parsen und in FullCalendar-Format umwandeln
    foreach ($results as $row) {
        // Liest den ICS-String und erstellt eine VObject-Struktur
        $vcalendar = VObject\Reader::read($row['calendardata']);
        
        // Holt das VEVENT-Objekt (Annahme: nur ein VEVENT pro calendarobject-Eintrag)
        $vevent = $vcalendar->VEVENT;
        
        if ($vevent) {
            
            // Bestimmt, ob es sich um ein ganztägiges Event handelt (DTSTART ohne Zeit)
            $isAllDay = isset($vevent->DTSTART) && $vevent->DTSTART->getDateType() === VObject\Property\DateTime::DATE;
            
            $eventData = [
                'id' => $row['id'], // Die DB-ID zur späteren Identifikation (wichtig für UPDATE/DELETE)
                'title' => (string)$vevent->SUMMARY,
                'start' => (string)$vevent->DTSTART, // VObject formatiert automatisch in ISO 8601
                'allDay' => $isAllDay
            ];
            
            // Fügen Sie das Enddatum nur hinzu, wenn es im ICS-Block vorhanden ist.
            if (isset($vevent->DTEND)) {
                 $eventData['end'] = (string)$vevent->DTEND;
            }

            $fullCalendarEvents[] = $eventData;
        }
    }

    // 6. JSON-Daten an das Frontend senden
    echo json_encode($fullCalendarEvents);

} catch (PDOException $e) {
    // Fehler bei der Datenbankverbindung/Abfrage
    http_response_code(500);
    // Nur generische Fehler im Produktivsystem ausgeben!
    echo json_encode(['error' => 'Datenbankfehler beim Laden der Kalenderdaten.']); 

} catch (\Exception $e) {
    // Fehler beim Parsen der ICS-Daten
    http_response_code(500);
    echo json_encode(['error' => 'Fehler beim Verarbeiten der Kalenderdaten: ' . $e->getMessage()]);
}

?>
