<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Familien-Kalender-Verwaltung 👨‍👩‍👧‍👦</title>
    
    <link href='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.11/main.min.css' rel='stylesheet' />
    <script src='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.11/index.global.min.js'></script>
    
    <style>
        /* Standard-Styling */
        body { 
            margin: 40px 10px; 
            padding: 0; 
            font-family: Arial, Helvetica, sans-serif; 
            background-color: #f7f7f7;
        }
        #calendar { 
            max-width: 1100px; 
            margin: 0 auto; 
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
    </style>
</head>
<body>

<div id='calendar'></div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var calendarEl = document.getElementById('calendar');

    // ----------------------------------------------------
    // HILFSFUNKTION: AJAX-Kommunikation mit dem Backend
    // ----------------------------------------------------
    
    /**
     * Sendet die Event-Daten und die gewünschte Aktion an das PHP-Backend.
     * @param {object} eventData - Enthält id, action, title, start, end, allDay.
     * @param {string} action - 'insert', 'update' oder 'delete'.
     * @param {object} fullCalendarInstance - Die Instanz des Kalenders (für addEvent).
     * @param {function} [revertFunction=null] - Optional: Funktion zum Rückgängigmachen (bei eventDrop).
     */
    function sendAction(eventData, action, fullCalendarInstance, revertFunction = null) {
        
        // Fügt die action-Property zur payload hinzu, falls sie fehlt (z.B. beim Löschen)
        eventData.action = action; 

        fetch('update_event.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(eventData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                console.log("Aktion erfolgreich: " + data.action);
                
                if (action === 'insert' && data.newId) {
                    // Fügt bei Erfolg das neu erstellte Event mit der korrekten DB-ID zum Frontend hinzu
                    fullCalendarInstance.addEvent({
                        id: data.newId, 
                        title: eventData.title,
                        start: eventData.start,
                        end: eventData.end,
                        allDay: eventData.allDay
                    });
                }
                
                alert('Termin erfolgreich ' + (action === 'insert' ? 'erstellt!' : data.action + '!'));

            } else {
                alert('Fehler beim Speichern des Termins: ' + data.message);
                if (revertFunction) { 
                    revertFunction(); // Macht die Änderung im Frontend rückgängig
                }
            }
        })
        .catch(error => {
            alert('Kommunikationsfehler: Konnte Server nicht erreichen.');
            if (revertFunction) { 
                revertFunction(); 
            }
            console.error('Error:', error);
        });
    }


    // ----------------------------------------------------
    // FULLCALENDAR INITIALISIERUNG
    // ----------------------------------------------------
    var calendar = new FullCalendar.Calendar(calendarEl, {
        initialView: 'dayGridMonth',
        locale: 'de', // Deutsche Sprache
        
        // Toolbar-Konfiguration
        headerToolbar: {
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,timeGridDay'
        },
        
        editable: true,    // Erlaubt Drag & Drop von Events
        selectable: true,  // Erlaubt das Auswählen von Zeiträumen (neue Events)
        timeZone: 'Europe/Berlin', // Wichtig für korrekte Zeitzonen-Verarbeitung

        // Datenquelle: Lädt Events über AJAX von get_events.php
        events: 'get_events.php', 

        // 1. Handler für das Verschieben/Ändern der Dauer (eventDrop)
        eventDrop: function(info) {
            let eventData = {
                id: info.event.id,
                title: info.event.title,
                start: info.event.start ? info.event.start.toISOString() : null,
                end: info.event.end ? info.event.end.toISOString() : null,
                allDay: info.event.allDay 
            };
            
            if (confirm("Sind Sie sicher, dass Sie '" + info.event.title + "' verschieben möchten?")) {
                sendAction(eventData, 'update', calendar, info.revert); 
            } else {
                info.revert(); 
            }
        },

        // 2. Handler für das Erstellen eines Events (select)
        select: function(info) {
            let title = prompt('Titel für neues Event eingeben:');
            
            if (title) {
                let eventData = {
                    title: title,
                    start: info.startStr, // Startdatum/zeit aus der Auswahl (ISO-Format)
                    end: info.endStr,     // Enddatum/zeit aus der Auswahl (ISO-Format)
                    allDay: info.allDay   
                };
                // Sendet die Daten zur Erstellung an das Backend
                sendAction(eventData, 'insert', calendar); 
            }
            calendar.unselect(); // Auswahl aufheben
        },
        
        // 3. Handler für das Bearbeiten des Titels und Löschen (eventClick)
        eventClick: function(info) {
            
            // --- A) Bearbeiten des Titels ---
            let newTitle = prompt('Neuen Titel für Event eingeben:', info.event.title);
            
            if (newTitle === null) {
                return; // Abbruch
            }
            
            if (newTitle !== info.event.title) {
                // Titel im Frontend aktualisieren
                info.event.setProp('title', newTitle); 
                
                let eventData = {
                    id: info.event.id,
                    title: newTitle,
                    start: info.event.start.toISOString(),
                    end: info.event.end ? info.event.end.toISOString() : null,
                    allDay: info.event.allDay 
                };
                // Datenbank aktualisieren
                sendAction(eventData, 'update', calendar);
            }

            // --- B) Lösch-Funktion ---
            if (confirm("Möchten Sie den Event '" + newTitle + "' wirklich löschen?")) {
                let deleteData = {
                    id: info.event.id,
                    title: info.event.title // Optional zur Protokollierung
                };
                sendAction(deleteData, 'delete', calendar);
                info.event.remove(); // Event aus der Frontend-Ansicht entfernen
            }
        }
    });

    calendar.render();
});
</script>

</body>
</html>
